//----------------------------------------------------------------------------------------------------
//
//   PWAVplayer
//
//   A polyphone WAV player based on ESP32
//
//   (C) 2025 colrhon.org
//   This program is released under the Creative Commons Public License, CC BY-NC-SA 4.0
// 
//   Firmware development with IDF-ESP, hardware is LilyGo TTGO T8 (ESP32-WROVER-E) Board, V1.8
//
//   Some terminology used:
//   A sound file is a file containing a header and audio data.
//   A track is a sound file being currently played, possibly together with other tracks.
//   The mixer takes of each track the foremost sound sample and mixes them to become a DAC value.
//   The fifo-buffer holds DAC values; it is being fed by the mixer while being unfed by the feeder.
//   The feeder periodically sends a DAC value to the DAC.
//

#include <stdio.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/stream_buffer.h"
#include <esp_system.h>
#include "esp_log.h"
#include "driver/sdmmc_host.h"
#include "driver/sdmmc_defs.h"
#include "driver/gptimer.h"
#include "driver/gpio.h"
#include "driver/sdspi_host.h"
#include "sdmmc_cmd.h"
#include "esp_vfs_fat.h"
#include "esp_check.h"
#include "esp_cpu.h"
#include "soc/gpio_struct.h"
#include "rom/ets_sys.h"
#include "esp_app_desc.h"

// adjust
//#define DEBUG
#include "pwav.h"

// Version
#define VERSION "0.9.1" 
char *gversion = VERSION;

// Firmware file name
#define FIRMWARE_NAME "update.bin"

// Local defines
#define MAX_INT16 (32767)
#define MIN_INT16 (-32768)
#define MAX_UINT16 (65535)
#define FPATHLEN 120
#define FAT_MAX_FILES 12  // max number of open files, see mount configuration for FatFS

// interprocess communication between WAVPlayer and PinEvents
#define IP_BUF_SZ 8
StreamBufferHandle_t xpinevt;

static char *glogmk = "WAV";
#define LOG glogmk

// Statistics
static struct stac {
    uint32_t isrodac; // FifoOut delivered a value
    uint32_t isrmiss; // FifoOut did not deliver a value
    uint32_t mixxcnt; // Mixer mixed at least 1 sample
} stac;

// Config data
#define CONFIG_NAME "config.txt"
#define CONF_DAC        0
#define CONF_DAC_12         1   // dac=12
#define CONF_DAC_16         2   // dac=16
#define CONF_MIX        1
#define CONF_MIX_SUM        1   // mix=sum
#define CONF_MIX_DIV2       2   // mix=div2
#define CONF_MIX_SQRT       3   // mix=sqrt
#define CONF_EVT        2
#define CONF_EVT_NONE       1   // evt=none
#define CONF_EVT_FLAT       2   // evt=flat
#define CONF_EVT_BW11       3   // evt=bw11
#define CONF_EVT_BG80       4   // evt=bg80
#define CONF_DEB        3       // debounce in ms
#define CONF_SER        4
#define CONF_SER_NONE       1   // ser=none
#define CONF_SER_I2C        2   // ser=i2c
#define CONF_SER_UART       3   // ser=uart
#define CONF_I2C_ADDR   5       // I2C slave address
#define CONF_MAX        6
static uint16_t gconf[CONF_MAX];

// Sound file
typedef struct sfile {
    struct sfile *next;
    uint16_t id;
    uint8_t attr[4];
    uint16_t vol;
    char fpath[FPATHLEN+1];
} Sfile;

// Running track
#define BUFSZ 32 // best buffer size: 16 < BUFSZ < 100
#define NMODES 1
typedef struct track {
    struct track *next;
    int fh;         // file handle
    uint32_t tcnt;  // total samples
    uint32_t rcnt;  // samples remaining
    uint32_t stpos; // start position of pcm data in file
    int16_t buf[BUFSZ];
    uint16_t bufl;  // buffer fill level
    uint16_t bidx;  // buffer index 
    Sfile *sf;
    char mode[NMODES]; // 0=delete
} Track;


// VFS FAT32
const char mount_point[] = "/sdcard";


//----------------------------------------------------------------------------------------
//
// Configuration data
//

// read config file and override default values
//
static void ReadConfig(char *fname) {
    FILE *fp;
    if ((fp = fopen(fname,"r")) == NULL) {
        return;
        ESP_LOGI(LOG, "No config found");
    }
#   define LBUF_SZ 100 
    char lbuf[LBUF_SZ];
    while (1) {
        if (fgets(lbuf,LBUF_SZ,fp) == NULL) break;
        char key[30];
        char val[30];
        int k = sscanf(lbuf," %[a-zA-Z0-9] = %[a-zA-Z0-9]",key,val);
        if (k == 2) {
            if (strcmp(key,"dac") == 0) {
                if (strcmp(val,"12") == 0) gconf[CONF_DAC] = CONF_DAC_12;
                if (strcmp(val,"16") == 0) gconf[CONF_DAC] = CONF_DAC_16;
            }
            if (strcmp(key,"mix") == 0) {
                if (strcmp(val,"sum") == 0) gconf[CONF_MIX] = CONF_MIX_SUM;
                if (strcmp(val,"div2") == 0) gconf[CONF_MIX] = CONF_MIX_DIV2;
                if (strcmp(val,"sqrt") == 0) gconf[CONF_MIX] = CONF_MIX_SQRT;
            }
            if (strcmp(key,"evt") == 0) {
                if (strcmp(val,"none") == 0) gconf[CONF_EVT] = CONF_EVT_NONE;
                if (strcmp(val,"flat") == 0) gconf[CONF_EVT] = CONF_EVT_FLAT;
                if (strcmp(val,"bw11") == 0) gconf[CONF_EVT] = CONF_EVT_BW11;
                if (strcmp(val,"bg80") == 0) gconf[CONF_EVT] = CONF_EVT_BG80;
            }
            if (strcmp(key,"deb") == 0) {
                gconf[CONF_DEB] = atoi(val);
            }
            if (strcmp(key,"ser") == 0) {
                if (strcmp(val,"none") == 0) gconf[CONF_SER] = CONF_SER_NONE;
                if (strcmp(val,"i2c") == 0) gconf[CONF_SER] = CONF_SER_I2C;
                if (strcmp(val,"uart") == 0) gconf[CONF_SER] = CONF_SER_UART;
            }
            if (strcmp(key,"addr") == 0) {
                gconf[CONF_I2C_ADDR] = (uint16_t)strtol(val,NULL,16);
            }
        }
    }
    fclose(fp);
}

static void InitConfig(void) {
    // set all defaults
    gconf[CONF_DAC] = CONF_DAC_12;
    gconf[CONF_MIX] = CONF_MIX_DIV2;
    gconf[CONF_EVT] = CONF_EVT_FLAT;
    gconf[CONF_DEB] = 5; // 5ms
    gconf[CONF_SER] = CONF_SER_NONE;
    gconf[CONF_I2C_ADDR] = 0x66;
}


//----------------------------------------------------------------------------------------
//
// WAV file
//

// Strip header from WAV file
// quick but not bombensicher
//

static int StripWavHeader(int fh, uint32_t *len, uint32_t *stpos) {
#   define QBUF 400
    char buf[QBUF];
    int n = read(fh,buf,QBUF);
    if (n != QBUF) return 1; // cannot read header
    if (strncmp(buf,"RIFF",4) != 0) return 2; // wrong file format
    if (strncmp(&buf[8],"WAVE",4) != 0) return 3; // wrong file content
    for (int i=0; i<QBUF; i++) {
        if ((buf[i]=='d') && (buf[i+1]=='a') && (buf[i+2]=='t') && (buf[i+3]=='a')) {
            uint32_t *p = (void *)&(buf[i+4]);
            *len = *p;
            lseek(fh,i+8,SEEK_SET);
            *stpos = i+8;
            return 0;
        }
    }
    return 4; // no 'data' chunk found
}


//----------------------------------------------------------------------------------------
//
// Chain of running tracks
//

static Track *tchain = NULL;

// Create and insert a new track
//
static Track *NewTrack(Sfile *s) {
    // open file
    int fh = open(s->fpath,O_RDONLY);
    if (fh < 0) {
        ESP_LOGE(LOG,"cannot open file %s",s->fpath);
        return NULL;
    }
    uint32_t tlen,stpos;
    int err = StripWavHeader(fh,&tlen,&stpos);
    if (err != 0) {
        ESP_LOGE(LOG,"not a valid WAV file, error %d -  %s",err,s->fpath);
        return NULL;
    }
    Track *t = (void *)malloc(sizeof(Track));
    t->next = tchain;
    tchain = t;
    t->fh = fh;
    t->tcnt = t->rcnt = tlen/2;
    t->stpos = stpos;
    t->bufl = 0;
    t->bidx = 0;
    t->sf = s; // refer soundfile
    for (int k = 0; k < NMODES; k++) t->mode[k] = 0;
    return t;
}

// Step through the track chain and remove all tracks which are marked for deletion
//
static void DeleteTracks(void) {
    for (Track **p = &tchain; *p != NULL; ) {
        Track *t = *p;
        if (t->mode[0] == 1) {
            close(t->fh);
            *p = t->next;
            free(t);
        }
        else {
            p = &(t->next);
        }
    }
}

// Mark tracks for deletion
//
static void MarkTracksById(int16_t id) {
    if (id < 0) { // mark all tracks
        for (Track *p = tchain; p != NULL; p = p->next) p->mode[0] = 1;
    }
    else { // mark tracks with given id
        for (Track *p = tchain; p != NULL; p = p->next) if (p->sf->id == (uint16_t)id) p->mode[0] = 1;
    }
}

static void PrintTracks(void) {
    ets_printf("Track list:\n");
    for (Track *p = tchain; p != NULL; p = p->next) {
        ets_printf("  Track: %s\n",p->sf->fpath);
    }
}

static int NoTrack(void) {
    return(tchain == NULL);
}


//----------------------------------------------------------------------------------------
//
// Fifo buffer
//

// Fifo is a ring buffer holding precalculated DAC values.
// For a smooth data stream to the DAC a buffer size of 1024 is required.
// 
//#define FIFO_SIZE 512 // must be a value 2^n (512, 1024, 2048, ..)
#define FIFO_SIZE 1024 // must be a value 2^n (512, 1024, 2048, ..)
#define FIFO_MASK (FIFO_SIZE-1)

struct Fifo {
    uint16_t data[FIFO_SIZE];
    uint16_t read;   // points to oldest entry
    uint16_t write;  // points to next empty slot
} fifo = {{}, 0, 0};

// Put value into the buffer, returns 0 if buffer is full
//
static inline uint8_t FifoIn(uint16_t mxval) {
    uint16_t next = ((fifo.write + 1) & FIFO_MASK);
    if (fifo.read == next) return 0;
    fifo.data[fifo.write] = mxval;
    fifo.write = next;
    return 1;
}

// Get a value from the buffer, return 0 if buffer empty
//
static inline uint8_t FifoOut(uint16_t *p) {
    if (fifo.read == fifo.write) return 0;
    *p = fifo.data[fifo.read];
    fifo.read = (fifo.read+1) & FIFO_MASK;
    return 1;
}

static inline uint8_t FifoFull() {
    uint16_t next = ((fifo.write + 1) & FIFO_MASK);
    return (fifo.read == next);
}

// Return the filling level of the buffer
//
static inline uint16_t FifoLevel(void) {
    if (fifo.write >= fifo.read) return fifo.write-fifo.read;
    else return FIFO_SIZE-(fifo.read-fifo.write);
}


//----------------------------------------------------------------------------------------
//
// Mixer
//

static int TryCloseTrack(Track *p) {
    if (p->sf->attr[0] == 'l') {
        // file marked with attribute 'l' (loop)
        lseek(p->fh,p->stpos,SEEK_SET);
        p->rcnt = p->tcnt;
        p->bufl = 0;
        p->bidx = 0;
        return 0;
    }
    // close down
    p->mode[0] = 1;
    return 1;
}

// Step through the track chain and mix one sample of each track
// There are different methods to mix, see configuration 'mix'
// Return 1 if chain contains an entry to be deleted, 0 otherwise
//
static uint8_t RunMixer() {

    if (FifoFull()) return 0; // nothing to do

    int cnt = 0;
    uint8_t touch = 1;
    uint8_t dflag = 0;
    int32_t tval = 0;
    for (Track *p = tchain; p != NULL; p = p->next) {
        if (p->mode[0]) dflag = 1;
        else if (p->rcnt == 0) dflag = TryCloseTrack(p);
        else {
            int32_t val = 0;
            if (p->bidx < p->bufl) {
                val = (int32_t)p->buf[p->bidx];
                p->bidx++;
                p->rcnt--;
            }
            else {
                // refill buffer from file
                ssize_t bcnt = read(p->fh,&(p->buf[0]),BUFSZ*sizeof(int16_t));
                if (bcnt == 0) { // eof
                    dflag = TryCloseTrack(p);
                }
                else if (bcnt < 0) { // read error
                    p->mode[0] = 1;
                    dflag = 1;
                }
                else { // continue
                    p->bufl = bcnt/sizeof(int16_t);
                    val = (int32_t)p->buf[0];
                    p->bidx = 1;
                    p->rcnt--;
                }
            }

            // adjust volume
            if (p->sf->vol != 100) val = (val * p->sf->vol)/100;

            // mixer method
            switch (gconf[CONF_MIX]) {
            case CONF_MIX_SUM:
                tval += val;
                break;
            case CONF_MIX_DIV2:
                tval += val/2;
                break;
            case CONF_MIX_SQRT:
                tval += val;
                cnt++;
                break;
            }

            // statistics
            if (touch) {
                stac.mixxcnt++;
                touch = 0;
            }
        }
    }

    // xxxxx may remove this mixing method
    if (gconf[CONF_MIX] == CONF_MIX_SQRT) {
        switch (cnt) {
        case 0: case 1:  // do nothing
            break;
        case 2:  // div by 1.4
            tval = (10*tval)/14;
            break;
        case 3:  // div by 1.7
            tval = (10*tval)/17;
            break;
        default: // div by 2
            tval = tval/2;
            break;
        }
    }
    
    // clip to int16_t range, add 1/2 range to make all positive,
    // then convert from int32 to uint16 and insert in Fifo
    //
    if (tval > MAX_INT16) tval = MAX_INT16; // clip volume
    if (tval < MIN_INT16) tval = MIN_INT16; // clip volume
    tval += -(MIN_INT16);
    FifoIn((uint16_t)tval);
    return dflag;
}

//
// Experimental
// Add a new track to the pre-calculated DAC values
// Accesses fifo-buffer w/out abstraction => needs improvement, tbd 
//
static void CorrFifo(Track *p) {
    uint16_t k;
    int16_t buf[FIFO_SIZE];

    if (p == NULL) return;
    k = FifoLevel();
    ssize_t bcnt = read(p->fh,&(buf[0]),k*sizeof(int16_t)); 
    int32_t tval;
    uint16_t j = fifo.read;
    for (uint16_t i = 0; i < bcnt/sizeof(uint16_t); i++) {
        tval = fifo.data[j];
        switch (gconf[CONF_MIX]) {
        case CONF_MIX_SUM:
            tval += buf[i];
            break;
        case CONF_MIX_DIV2:
            tval += buf[i]/2;
            break;
        case CONF_MIX_SQRT:
            tval += buf[i]/2; //cheating
            break;
        }
        if (tval > MAX_UINT16) tval = MAX_UINT16; // clip volume
        if (tval < 0) tval = 0; // clip volume
        fifo.data[j] = (uint16_t)tval;
        j = (j+1)&FIFO_MASK;
    }
    p->rcnt -= bcnt/sizeof(uint16_t);
}


//----------------------------------------------------------------------------------------
//
// DAC
// Configure internal DAC of EPS32
// Configure SPI bus and add external DAC as device
//

#define PMOSI GPIO_NUM_23
#define PSCLK GPIO_NUM_18
#define PPCS   GPIO_NUM_5

#if 1 // bitbang version, SPI version see below
//
// The bitbang version is much faster then the SPI bus version, see below
// Duration of interrupt: 2.7us (vs 12.5us with SPI)
//

static void InitExtDAC(void) {

    // configure GPIO for output
    gpio_config_t io_conf = {0};
    io_conf.intr_type = GPIO_INTR_DISABLE;
    io_conf.mode = GPIO_MODE_OUTPUT;
    io_conf.pin_bit_mask = (1ULL << PMOSI)|(1ULL << PSCLK)|(1ULL << PPCS);
    io_conf.pull_down_en = GPIO_PULLDOWN_DISABLE;
    io_conf.pull_up_en = GPIO_PULLUP_DISABLE; // remove
    ESP_ERROR_CHECK(gpio_config(&io_conf));

    gpio_set_level(PPCS,1);
    gpio_set_level(PMOSI,0);
    gpio_set_level(PSCLK,0);
}

static inline void LoadExtDAC(uint16_t val) {

//  TP_SET();
    // 12bit, DAC MCP4821
#   define DAC_CONFIG_BITS 0x3000;
    if (gconf[CONF_DAC] == CONF_DAC_12) val = (val>>4) | DAC_CONFIG_BITS;
    
    GPIO.out_w1tc = 1UL << PPCS; // CS low
    for (uint8_t i=0; i<16; i++) { 
        if (val & 0x8000) GPIO.out_w1ts = 1UL << PMOSI;
        else GPIO.out_w1tc = 1UL << PMOSI;
        GPIO.out_w1ts = 1UL << PSCLK;
        GPIO.out_w1tc = 1UL << PSCLK;
        val = val<<1;
    }
    GPIO.out_w1ts = 1UL << PPCS; // CS high
    GPIO.out_w1tc = 1UL << PMOSI;
    GPIO.out_w1tc = 1UL << PSCLK;
//  TP_CLR();
}

#else // SPI version
//
// data transfer with SPI3 and spi_device_polling_transmit()
// 
// not used, because transmitting small data chunks consumes too much time
// initiating the transfer with spi_device_polling_transmit() takes 8us,
// exit (after transfer) another 3us,
// a total of 11us overhead, see scope screenshot sd-01.png
//
// see https://esp32.com/viewtopic.php?t=10546
// see https://esp32.com/viewtopic.php?t=24774
// see https://esp32.com/viewtopic.php?t=25417
// see https://esp32.com/viewtopic.php?t=8720
// all same problem, no solution
// 

spi_device_handle_t spi_device;

// configure SPI3 host (VSPI) for external DAC
//
static void InitExtDAC(void) {

    ESP_LOGI(LOG,"Initializing SPI bus...");
    // configuration for the SPI bus
    spi_bus_config_t buscfg = {
        .mosi_io_num = PMOSI,
        .miso_io_num = -1,   // not used
        .sclk_io_num = PSCLK,
        .quadwp_io_num = -1, // not used
        .quadhd_io_num = -1,  // not used
        .max_transfer_sz = 32 // xxxx Max transfer size in bytes
    };

    // initialize the SPI bus
    ESP_ERROR_CHECK(spi_bus_initialize(SPI3_HOST,&buscfg,SPI_DMA_DISABLED));
    ESP_LOGI(LOG,"SPI bus initialized.");

    ESP_LOGI(LOG,"Adding SPI device...");
    // configuration for the DAC MCP4821
    spi_device_interface_config_t devcfg = {0};
    devcfg.clock_speed_hz = 10 * 1000 * 1000; // clock out @ 10MHz (up to 20MHz)
    devcfg.mode = 0;                          // SPI mode 0 (CPOL=0, CPHA=0)
    devcfg.spics_io_num = PPCS;               // CS pin
    devcfg.queue_size = 1;                    // xxxxx

    // add device
    ESP_ERROR_CHECK(spi_bus_add_device(SPI3_HOST,&devcfg,&spi_device));
    ESP_LOGI(LOG,"SPI device added.");
}

static inline void LoadExtDAC(uint16_t val) {
#   define DAC_CONFIG_BITS 0x30
    uint8_t sbuf[2];
//  TP_SET();
    sbuf[0] = val&0xff;
    // 12bit, DAC MCP4821
    if (gconf[CONF_DAC] == CONF_DAC_12) sbuf[1] = ((val>>8) & 0x0f) + DAC_CONFIG_BITS;
    else sbuf[1] = val>>8;
    spi_transaction_t t;
    memset(&t,0,sizeof(t)); // zero out the transaction structure
    t.length = 16;
    t.tx_buffer = sbuf;
    ESP_ERROR_CHECK(spi_device_polling_transmit(spi_device,&t));
//  TP_CLR();
}

#endif

//----------------------------------------------------------------------------------------
//
// DAC Feeder (ISR)
// Calling period 22.7us (44.1kHz)
// In case fifo-buffer is empty, simply leave DAC to hold its current value 1 tick longer

static bool LoadDAC_cb(gptimer_handle_t timer, const gptimer_alarm_event_data_t *edata, void *user_ctx) {
    uint16_t mxval = 0;
    if (FifoOut(&mxval)) {
        LoadExtDAC(mxval);
        stac.isrodac++;
    }
    else stac.isrmiss++;
    return false;
}

// Setup periodic interrupt @ 44.1 kHz
//
static void SetupDACFeeder(void) {

    gptimer_handle_t gptimer = NULL;
    gptimer_config_t timer_config = {
        .clk_src = GPTIMER_CLK_SRC_DEFAULT, // select the default clock source
        .direction = GPTIMER_COUNT_UP,      // counting direction is up
//        .resolution_hz = 1 * 1000 * 1000,   // resolution is 1 MHz, i.e., 1 tick equals 1 microsecond
        .resolution_hz = 1 * 1000 * 1058,   // 1 tick equals 0.9452 microsecond
    };
    // create a timer instance
    ESP_ERROR_CHECK(gptimer_new_timer(&timer_config, &gptimer));
    

    gptimer_alarm_config_t alarm_config = {
        .reload_count = 0,      // when the alarm event occurs, the timer will automatically reload to 0
        .alarm_count = 24,      // 44.1 kHz @ res 0.9452 us
        .flags.auto_reload_on_alarm = true, // Enable auto-reload function
    };
    // set the timer's alarm action
    ESP_ERROR_CHECK(gptimer_set_alarm_action(gptimer, &alarm_config));

    gptimer_event_callbacks_t cbs = {
        .on_alarm = LoadDAC_cb, // user callback function
        };
    // egister timer event callback functions
    ESP_ERROR_CHECK(gptimer_register_event_callbacks(gptimer, &cbs, NULL));
    ESP_ERROR_CHECK(gptimer_enable(gptimer));
    ESP_ERROR_CHECK(gptimer_start(gptimer));
}


//----------------------------------------------------------------------------------------
//
// SDMMC
// mount SD card
//

static sdmmc_card_t *card;

static esp_err_t MountSDCard(void) {

    sdmmc_host_t host = SDMMC_HOST_DEFAULT();
    host.max_freq_khz = SDMMC_FREQ_HIGHSPEED; // 40MHz; disable to reduce to 20MHz

    sdmmc_slot_config_t slot = SDMMC_SLOT_CONFIG_DEFAULT();
    slot.width = 4;
    slot.d1 = GPIO_NUM_4;
    slot.d2 = GPIO_NUM_12;
    slot.d3 = GPIO_NUM_13;    
    slot.flags |= SDMMC_SLOT_FLAG_INTERNAL_PULLUP;

    esp_vfs_fat_sdmmc_mount_config_t mount = {
        .format_if_mount_failed = false,
        .max_files = FAT_MAX_FILES,
    };

    esp_err_t err = esp_vfs_fat_sdmmc_mount(mount_point, &host, &slot, &mount, &card);
    if (err != ESP_OK) {
        ESP_LOGE(LOG,"SD card, mount failed: %s", esp_err_to_name(err));
        ESP_LOGE(LOG,"Check whether SD card is inserted");
        return err;
        }

    // print card properties
    sdmmc_card_print_info(stdout, card);
    
    return ESP_OK;
}


//----------------------------------------------------------------------------------------
//
// Read catalog of titles from SD
// Start playing sound file
//
// Example for the structure of the name of a sound file:
// 0012-xbxx-100-ring-my-bell.wav 
//

// chain of all sound files
static Sfile *fchain = NULL;

// initial start of sound files marked with attribute 'i'
//
static void StartInitSfiles() {
    for (Sfile *s = fchain; s != NULL; s = s->next) if (s->attr[2] == 'i') NewTrack(s);
}

static Sfile *SearchSfile(uint16_t id) {
    for (Sfile *s = fchain; s != NULL; s = s->next) if (s->id == id) return s; 
    return NULL;
}

static void PrintSfiles(void) {
    ets_printf("Soundfile list:\n");
    for (Sfile *s = fchain; s != NULL; s = s->next) {
        ets_printf("  Sound: %s\n",s->fpath);
    }
}

// Insert sound as a track into the track list
// Abort silently if something goes wrong
//
static void StartSfile(uint16_t id) {
    Sfile *s = SearchSfile(id);
    if (s == NULL) return;
    if (s->attr[3] == 'k') { // kill
        MarkTracksById(-1);
        NewTrack(s);
    }
    else if (s->attr[1] == 'b') { // break
        MarkTracksById((int16_t)id);
        NewTrack(s);
    }
    else {
//        NewTrack(s);
        CorrFifo(NewTrack(s));
    }
}

// Create a sfile entry
//
static int16_t InsertFlist(char *fpath, char *fname) {
    char buf[200];
    strcpy(buf,fpath);
    strcat(buf,fname);
    ESP_LOGI(LOG,"consider file %s",buf);
    Sfile *s = (void *)malloc(sizeof(Sfile));
    if (s == NULL) {
        ESP_LOGI(LOG,"malloc failed");
        return -1;
        }
    int n = sscanf(fname,"%hu-%c%c%c%c-%hu-%*s",
                   &(s->id),&(s->attr[0]),&(s->attr[1]),&(s->attr[2]),&(s->attr[3]),&(s->vol));
    if (n != 6) {
        // ESP_LOGI(LOG,"discarded %s",fname);
        free(s);
        return -1;
    }
    if (s->vol > 100) s->vol = 100;
    strncpy(s->fpath,buf,FPATHLEN);
    s->fpath[FPATHLEN] = 0;
    s->next = fchain;
    fchain = s;
    return s->id;
}

// Create an ordinary sfile entry containing the spoken version 
// It's ID is 11111
//
static int16_t InsertFlistVersion(char *fname) {
    ESP_LOGI(LOG,"insert version file %s",fname);
    Sfile *s = (void *)malloc(sizeof(Sfile));
    if (s == NULL) {
        ESP_LOGI(LOG,"malloc failed");
        return -1;
        }
    s->id = 11111;
    s->attr[0] = 'x';
    s->attr[1] = 'x';
    s->attr[2] = 'x';
    s->attr[3] = 'x';
    s->vol = 100;
    strncpy(s->fpath,fname,FPATHLEN);
    s->fpath[FPATHLEN] = 0;
    s->next = fchain;
    fchain = s;
    return s->id;
}

static void ReadCatalogFromSD() {
    char fpath0[30];
    sprintf(fpath0, "%s/",mount_point);
    ESP_LOGI(LOG,"read catalog %s",fpath0);
    uint16_t ne = 0;
    DIR *dp;
    struct dirent *ep;
    dp = opendir (fpath0);
    if (dp != NULL) {
        while ((ep = readdir(dp)) != NULL) {
            if (InsertFlist(fpath0,ep->d_name) < 0) continue;
            ne++;
        }
        closedir(dp);
        ESP_LOGI(LOG,"read %u titles",ne);
    }
    else {
        ESP_LOGI(LOG,"read failed %s",fpath0);
    }
}


//----------------------------------------------------------------------------------------
//
// WAV Player
//
//

static void ExecCommand(Rxcmd *k) {
    switch (k->cmd) {
    case 'p': // play sound
        StartSfile(k->arg);
        break;
    case 's': // tbd, stop sound
        break;
    case 'q': // tbd, queue sound
        break;
    case 'v': // tbd, set volume of sound
        break;
    default:
        break;
    }
}

static void WAVPlayer(void *pvParameters) {

    // configure test & LED pin
    gpio_config_t io_conf = {0};
    io_conf.intr_type = GPIO_INTR_DISABLE;
    io_conf.mode = GPIO_MODE_OUTPUT;
    io_conf.pin_bit_mask = (1ULL << TESTPIN);
    io_conf.pull_down_en = GPIO_PULLDOWN_DISABLE;
    io_conf.pull_up_en = GPIO_PULLUP_DISABLE;
    ESP_ERROR_CHECK(gpio_config(&io_conf));

    char buf[100];
    const esp_app_desc_t *ppd = esp_app_get_description();
    printf("Version: %s\n",ppd->version);

    // spoken version
    sprintf(buf,"%s/spokenvers/version-%c-%c-%c.wav",mount_point,ppd->version[0],ppd->version[2],ppd->version[4]);
    int fh = open(buf,O_RDONLY);
    if (fh >= 0) {
        close(fh);
        InsertFlistVersion(buf);
        StartSfile(11111);
    }
    
    ReadCatalogFromSD();
    InitExtDAC();
    SetupDACFeeder();

    // play spoken version if linked in
    uint8_t run = 1;
    while (run) {
        if (RunMixer()) DeleteTracks();
        if (FifoFull()) vTaskDelay(1);
        if (NoTrack()) run = 0;
    }
    
#ifdef DEBUG
    //StartSfile(16); // you want to play, let's play 'i'
    //StartSfile(18); // horn, loop 'l'
    //StartSfile(5);  // kill 'k'
    //StartSfile(6);  // break 'b'
    //StartSfile(7);
    //StartSfile(50);  // Fred Wesley, House Party
    //StartSfile(101);  // Sinus 1kHz
    //PrintTracks();
#endif

    // statistics
    stac.isrmiss = 0;
    stac.isrodac = 0;
    stac.mixxcnt = 0;
    StartInitSfiles();

    Rxcmd xcmd;
    run = 1;
    while (run) {
        if (RunMixer()) DeleteTracks();
        if (FifoFull()) vTaskDelay(1);
        if (xStreamBufferReceive(xpinevt,&xcmd,sizeof(Rxcmd),0) == sizeof(Rxcmd)) ExecCommand(&xcmd);
        
#ifdef DEBUG
        if (stac.mixxcnt == 200000) StartSfile(1);
        if (stac.mixxcnt == 280000) StartSfile(2);
        if (stac.mixxcnt == 300000) StartSfile(3);
        if (stac.mixxcnt == 320000) StartSfile(4);
        if (stac.mixxcnt == 600000) StartSfile(5);
        //if ((stac.mixxcnt) > 650000) run=0; // exit
        //if (NoTrack()) run=0; // exit after last track ends
#endif
        
    }
    
}

static void PrintStatistics(void) {
    ets_printf("Statistik:\n");
    ets_printf("  ISR odac: %lu (total von Fifo geliefert und an DAC gesendete Werte)\n", stac.isrodac);
    ets_printf("  ISR miss: %lu (total Anzahl Fifo leer)\n", stac.isrmiss);
    ets_printf("  Mix xcnt: %lu (total von Mixer gemischte Werte mit mind. 1 Track\n", stac.mixxcnt);
    ets_printf("  odac + miss = total DAC Feeder Aufrufe\n");
}

static void WAVDummy(void *pvParameters) {
    const esp_app_desc_t *ppd = esp_app_get_description();
    printf("Version: %s\n",ppd->version);
    int run = 1;
    while (run) {
        vTaskDelay(100);
    }
}


//----------------------------------------------------------------------------------------
//
// Main
//
//

#define CORE_0 0
#define CORE_1 1

extern void SerialUART(void *pvParameters);
extern void SerialI2C(void *pvParameters);
extern void PinEvents(void *pvParameters);
extern void EncEventW11(void *pvParameters);
extern void EncEventG80(void *pvParameters);
extern void CheckFWUpdate(char *fname);


void app_main(void) {
    char fpath0[30];

    if (MountSDCard() != ESP_OK) return;

    sprintf(fpath0, "%s/%s",mount_point,FIRMWARE_NAME);
    CheckFWUpdate(fpath0);
    
    InitConfig();
    sprintf(fpath0, "%s/%s",mount_point,CONFIG_NAME);
    ReadConfig(fpath0);

    xpinevt = xStreamBufferCreate(IP_BUF_SZ,1);
    xTaskCreatePinnedToCore(&WAVPlayer, "WAVplayer", 4096, NULL, tskIDLE_PRIORITY+10, NULL, CORE_0);
//    xTaskCreatePinnedToCore(&WAVDummy, "WAVdummy", 4096, NULL, tskIDLE_PRIORITY+10, NULL, CORE_0);

    switch (gconf[CONF_SER]) {
    case CONF_SER_UART:
        xTaskCreatePinnedToCore(&SerialUART, "SerialUART", 4096, NULL, (tskIDLE_PRIORITY + 2), NULL, CORE_1);
        break;
    case CONF_SER_I2C:
        xTaskCreatePinnedToCore(&SerialI2C, "SerialI2C", 4096, &(gconf[CONF_I2C_ADDR]), (tskIDLE_PRIORITY + 2), NULL, CORE_1);
        break;
    default: break;
    }

    switch(gconf[CONF_EVT]) {
    case CONF_EVT_FLAT:
        xTaskCreatePinnedToCore(&PinEvents, "PinEvents", 4096, &(gconf[CONF_DEB]), (tskIDLE_PRIORITY + 3), NULL, CORE_1);
        break;
    case CONF_EVT_BW11:
        xTaskCreatePinnedToCore(&EncEventW11, "EncEventW11", 4096, NULL, (tskIDLE_PRIORITY + 3), NULL, CORE_1);
        break;
    case CONF_EVT_BG80:
        xTaskCreatePinnedToCore(&EncEventG80, "EncEventG80", 4096, NULL, (tskIDLE_PRIORITY + 3), NULL, CORE_1);
        break;
    default: break;
    }

}

